/**
 * Auto-Archiver
 * Automatically moves low-scoring tabs to vault and optionally suspends them
 */

import { storage } from './storage.js';
import { getArchivableTabs, getSuspendableTabs, THRESHOLDS } from './scoring.js';
import { getPersonalizationScore } from './patterns.js';
import { buildSearchIndex } from './search.js';

// Configuration
const SCREENSHOT_MAX_WIDTH = 800; // Limit screenshot size

/**
 * Main archiving function - checks and archives eligible tabs
 */
export async function checkAndArchiveTabs() {
  try {
    // Get max active tabs from settings
    const maxActiveTabs = await storage.getSetting('maxActiveTabs', 20);
    
    // Get all open tabs
    const openTabs = await chrome.tabs.query({});
    
    console.log(`Tab count: ${openTabs.length}, Max active: ${maxActiveTabs}`);
    
    // Check if we should trigger archiving
    const shouldArchive = openTabs.length > maxActiveTabs;
    
    if (!shouldArchive) {
      console.log('Tab count below threshold, skipping archive');
      return;
    }
    
    // Get tabs eligible for archiving
    const archivable = await getArchivableTabs();
    
    console.log(`Found ${archivable.length} archivable tabs`);
    
    if (archivable.length === 0) {
      console.log('No tabs meet archiving criteria (score below threshold)');
      return;
    }
    
    for (const tabData of archivable) {
      await archiveTab(tabData.tabId);
    }
    
    // Handle suspension separately for very low scores
    await suspendLowScoreTabs();
    
  } catch (error) {
    console.error('Archive check failed:', error);
  }
}

/**
 * Archive a single tab
 */
async function archiveTab(tabId) {
  try {
    // Get archive threshold from settings
    const archiveThreshold = await storage.getSetting('archiveThreshold', THRESHOLDS.ARCHIVE);
    const minInactiveMinutes = await storage.getSetting('minInactiveMinutes', 30);
    
    // Get tab info
    const tab = await chrome.tabs.get(tabId);
    if (!tab) {
      console.log(`Tab ${tabId} not found`);
      return;
    }
    
    // Skip if tab is active or pinned
    if (tab.active || tab.pinned) {
      console.log(`Skipping tab ${tabId} (${tab.active ? 'active' : 'pinned'})`);
      return;
    }
    
    // Get tab data and signals
    const tabData = await storage.getTab(tabId);
    if (!tabData) {
      console.log(`No data for tab ${tabId}`);
      return;
    }
    
    // Check if tab has been inactive long enough
    const inactiveTimeMs = Date.now() - (tabData.lastActivated || Date.now());
    const minInactiveMs = minInactiveMinutes * 60 * 1000;
    
    if (inactiveTimeMs < minInactiveMs) {
      const remainingMinutes = Math.ceil((minInactiveMs - inactiveTimeMs) / 60000);
      console.log(`Tab ${tabId} not inactive long enough (${remainingMinutes}m remaining)`);
      return;
    }
    
    // Check domain rules
    try {
      const domain = new URL(tab.url).hostname;
      const domainRules = await storage.getSetting('domainRules', []);
      const matchingRule = domainRules.find(rule => 
        domain === rule.domain || domain.endsWith('.' + rule.domain)
      );
      
      if (matchingRule) {
        if (matchingRule.type === 'never') {
          console.log(`Tab ${tabId} skipped - never archive rule for ${domain}`);
          return;
        } else if (matchingRule.type === 'threshold') {
          // Override archive threshold for this domain
          if (tabData.score >= matchingRule.value) {
            console.log(`Tab ${tabId} score ${tabData.score} >= domain threshold ${matchingRule.value}`);
            return;
          }
        } else if (matchingRule.type === 'inactive') {
          // Override inactive time for this domain
          const domainInactiveMs = matchingRule.value * 60 * 1000;
          if (inactiveTimeMs < domainInactiveMs) {
            const remaining = Math.ceil((domainInactiveMs - inactiveTimeMs) / 60000);
            console.log(`Tab ${tabId} not inactive long enough for domain rule (${remaining}m remaining)`);
            return;
          }
        }
      }
      
      // ML Personalization Check
      const enableML = await storage.getSetting('enableMLPersonalization', true);
      if (enableML) {
        const mlPrediction = await getPersonalizationScore(domain);
        
        if (mlPrediction.shouldProtect) {
          console.log(`ML: Protecting ${domain} - ${mlPrediction.reason}`);
          return;  // Don't archive
        }
        
        if (mlPrediction.scoreBoost > 0) {
          tabData.score += mlPrediction.scoreBoost;
          console.log(`ML: Boosting ${domain} score by ${mlPrediction.scoreBoost} - ${mlPrediction.reason}`);
        }
      }
    } catch (e) {
      // Invalid URL, continue with normal archiving
    }
    
    // Skip if score is above threshold (safety check)
    if (tabData.score >= archiveThreshold) {
      console.log(`Tab ${tabId} score ${tabData.score} >= threshold ${archiveThreshold}`);
      return;
    }
    
    // Capture screenshot
    let screenshot = null;
    try {
      // Make tab active temporarily to capture
      await chrome.tabs.update(tabId, { active: true });
      await new Promise(resolve => setTimeout(resolve, 100)); // Wait for render
      
      screenshot = await chrome.tabs.captureVisibleTab(tab.windowId, {
        format: 'png'
      });
      
    } catch (error) {
      console.warn('Screenshot capture failed:', error);
    }
    
    // Extract domain from URL
    let domain = '';
    try {
      const url = new URL(tab.url);
      domain = url.hostname;
    } catch (error) {
      domain = tab.url;
    }
    
    // Get latest signals for text extract and scroll position
    const signals = await storage.getSignalsForTab(tabId);
    const latestSignal = signals[signals.length - 1] || {};
    
    // Create vault entry
    const vaultEntry = {
      url: tab.url,
      title: tab.title,
      domain,
      favIconUrl: tab.favIconUrl || '',
      textExtract: latestSignal.textExtract || '',
      screenshot,
      scrollPosition: latestSignal.scrollPosition || 0,
      score: tabData.score,
      archivedAt: Date.now(),
      originalTabId: tabId,
      signals: {
        dwellTime: latestSignal.dwellTime || 0,
        scrollDepth: latestSignal.scrollDepth || 0,
        timeOnPage: latestSignal.timeOnPage || 0
      }
    };
    
    // Save to vault
    const vaultId = await storage.addVaultEntry(vaultEntry);
    
    // Build search index for TF-IDF
    try {
      await buildSearchIndex(vaultId, vaultEntry);
    } catch (error) {
      console.error('Failed to build search index:', error);
    }
    
    console.log(`Archived tab ${tabId} (${tab.title}) to vault ${vaultId}`);
    
    // Close the archived tab
    try {
      await chrome.tabs.remove(tabId);
      console.log(`Closed archived tab ${tabId}`);
    } catch (error) {
      console.warn(`Failed to close tab ${tabId}:`, error);
    }
    
    // Clean up storage
    await storage.deleteTab(tabId);
    await storage.deleteSignalsForTab(tabId);
    
  } catch (error) {
    console.error(`Failed to archive tab ${tabId}:`, error);
  }
}

/**
 * Suspend tabs with very low scores to free memory
 */
async function suspendLowScoreTabs() {
  try {
    const suspendable = await getSuspendableTabs();
    
    for (const tabData of suspendable) {
      try {
        const tab = await chrome.tabs.get(tabData.tabId);
        
        // Skip active, pinned, or already discarded tabs
        if (tab.active || tab.pinned || tab.discarded) continue;
        
        // Discard tab to free memory (doesn't close it)
        await chrome.tabs.discard(tabData.tabId);
        
        console.log(`Suspended tab ${tabData.tabId} (${tab.title})`);
        
      } catch (error) {
        console.warn(`Failed to suspend tab ${tabData.tabId}:`, error);
      }
    }
    
  } catch (error) {
    console.error('Suspension failed:', error);
  }
}

/**
 * Manually archive a specific tab (called from UI)
 */
export async function manuallyArchiveTab(tabId) {
  await archiveTab(tabId);
}

/**
 * Update archiving threshold
 */
export function setMaxActiveTabs(count) {
  MAX_ACTIVE_TABS = count;
}
