/**
 * ML Pattern Detection Engine
 * Learns user browsing patterns from real data for personalized archiving
 */

import { storage } from './storage.js';

// Constants
const LEARNING_PERIOD_DAYS = 7;  // Start making predictions after 1 week
const STARTUP_DETECTION_WINDOW = 300000; // 5 minutes in ms
const MIN_STARTUP_FREQUENCY = 0.7; // 70% frequency to be considered routine

/**
 * Track a tab visit with real data
 */
export async function trackVisit(domain, duration, context = {}) {
  try {
    const pattern = await storage.getPattern(domain) || {
      domain,
      visitCount: 0,
      totalTimeSpent: 0,
      sessions: [],
      reopens: 0,
      startupAppearances: 0,
      hourDistribution: new Array(24).fill(0),
      dayDistribution: new Array(7).fill(0),
      lastVisit: null,
      firstSeen: Date.now(),
      importance: 0
    };
    
    // Update visit stats
    pattern.visitCount++;
    pattern.totalTimeSpent += duration;
    pattern.lastVisit = Date.now();
    
    // Update temporal patterns
    const now = new Date();
    const hour = now.getHours();
    const day = now.getDay();
    pattern.hourDistribution[hour]++;
    pattern.dayDistribution[day]++;
    
    // Store session
    pattern.sessions.push({
      timestamp: Date.now(),
      duration,
      hour,
      day,
      ...context
    });
    
    // Keep only last 100 sessions
    if (pattern.sessions.length > 100) {
      pattern.sessions = pattern.sessions.slice(-100);
    }
    
    // Calculate importance score
    pattern.importance = calculateImportance(pattern);
    
    await storage.savePattern(pattern);
    
  } catch (error) {
    console.error('Failed to track visit:', error);
  }
}

/**
 * Track tab restored from archive (indicates importance)
 */
export async function trackRestore(domain) {
  try {
    const pattern = await storage.getPattern(domain);
    if (pattern) {
      pattern.reopens++;
      pattern.importance = calculateImportance(pattern);
      await storage.savePattern(pattern);
    }
  } catch (error) {
    console.error('Failed to track restore:', error);
  }
}

/**
 * Track tabs opened on browser startup
 */
export async function trackStartup(domains) {
  try {
    const startupSession = {
      timestamp: Date.now(),
      domains,
      dayOfWeek: new Date().getDay(),
      hour: new Date().getHours()
    };
    
    // Save startup session
    const sessions = await storage.getStartupSessions() || [];
    sessions.push(startupSession);
    
    // Keep only last 50 startup sessions
    if (sessions.length > 50) {
      sessions.shift();
    }
    
    await storage.saveStartupSessions(sessions);
    
    // Update domain patterns
    for (const domain of domains) {
      const pattern = await storage.getPattern(domain);
      if (pattern) {
        pattern.startupAppearances++;
        await storage.savePattern(pattern);
      }
    }
    
  } catch (error) {
    console.error('Failed to track startup:', error);
  }
}

/**
 * Calculate importance score (0-100) based on real usage data
 */
function calculateImportance(pattern) {
  const now = Date.now();
  const ageInDays = (now - pattern.firstSeen) / (1000 * 60 * 60 * 24);
  
  // Don't make strong predictions during learning period
  if (ageInDays < LEARNING_PERIOD_DAYS) {
    return 0;
  }
  
  // Get max values for normalization
  const maxVisits = 100;  // Normalize to 100 visits
  const maxTime = 3600000; // Normalize to 1 hour
  const maxReopens = 10;
  
  // Calculate component scores
  const frequencyScore = Math.min(pattern.visitCount / maxVisits, 1) * 30;
  const timeScore = Math.min(pattern.totalTimeSpent / maxTime, 1) * 25;
  const reopenScore = Math.min(pattern.reopens / maxReopens, 1) * 10;
  
  // Recency score (decay over time)
  const daysSinceVisit = pattern.lastVisit ? 
    (now - pattern.lastVisit) / (1000 * 60 * 60 * 24) : 999;
  const recencyScore = Math.max(0, (1 - daysSinceVisit / 30)) * 20;
  
  // Temporal relevance score
  const temporalScore = calculateTemporalRelevance(pattern) * 15;
  
  const total = frequencyScore + timeScore + reopenScore + recencyScore + temporalScore;
  
  return Math.min(Math.round(total), 100);
}

/**
 * Calculate temporal relevance (is this tab relevant NOW?)
 */
function calculateTemporalRelevance(pattern) {
  const now = new Date();
  const currentHour = now.getHours();
  const currentDay = now.getDay();
  
  // Get usage frequency for current hour and day
  const hourTotal = pattern.hourDistribution.reduce((a, b) => a + b, 0);
  const dayTotal = pattern.dayDistribution.reduce((a, b) => a + b, 0);
  
  if (hourTotal === 0 || dayTotal === 0) return 0;
  
  const hourMatch = pattern.hourDistribution[currentHour] / hourTotal;
  const dayMatch = pattern.dayDistribution[currentDay] / dayTotal;
  
  // Combine hour and day relevance
  return (hourMatch * 0.6 + dayMatch * 0.4);
}

/**
 * Get personalization prediction for a domain
 */
export async function getPersonalizationScore(domain) {
  try {
    const pattern = await storage.getPattern(domain);
    
    if (!pattern) {
      return {
        importance: 0,
        shouldProtect: false,
        scoreBoost: 0,
        reason: 'No pattern data'
      };
    }
    
    const importance = pattern.importance;
    
    // Protection threshold: 60+
    if (importance >= 60) {
      return {
        importance,
        shouldProtect: true,
        scoreBoost: 0,
        reason: `High importance (${importance}%) - frequently used pattern`
      };
    }
    
    // Boost threshold: 40-59
    if (importance >= 40) {
      const boost = Math.round((importance - 40) * 2); // 0-40 point boost
      return {
        importance,
        shouldProtect: false,
        scoreBoost: boost,
        reason: `Moderate importance (${importance}%) - boost score by ${boost}`
      };
    }
    
    // Normal archiving
    return {
      importance,
      shouldProtect: false,
      scoreBoost: 0,
      reason: `Low importance (${importance}%) - normal archiving`
    };
    
  } catch (error) {
    console.error('Failed to get personalization score:', error);
    return { importance: 0, shouldProtect: false, scoreBoost: 0, reason: 'Error' };
  }
}

/**
 * Get predicted startup tabs based on learned patterns
 */
export async function getPredictedStartupTabs(context = {}) {
  try {
    const sessions = await storage.getStartupSessions() || [];
    
    if (sessions.length < 10) {
      return []; // Not enough data
    }
    
    const { dayOfWeek = new Date().getDay(), hour = new Date().getHours() } = context;
    
    // Count domain frequencies in startup sessions
    const domainCounts = {};
    let relevantSessions = 0;
    
    for (const session of sessions) {
      // Filter by context if similar time/day
      const isRelevant = 
        (session.dayOfWeek === dayOfWeek) ||
        (Math.abs(session.hour - hour) <= 2);
      
      if (isRelevant) {
        relevantSessions++;
        for (const domain of session.domains) {
          domainCounts[domain] = (domainCounts[domain] || 0) + 1;
        }
      }
    }
    
    if (relevantSessions === 0) return [];
    
    // Find domains that appear in >70% of relevant sessions
    const startupTabs = [];
    for (const [domain, count] of Object.entries(domainCounts)) {
      const frequency = count / relevantSessions;
      if (frequency >= MIN_STARTUP_FREQUENCY) {
        startupTabs.push({
          domain,
          frequency,
          confidence: Math.round(frequency * 100)
        });
      }
    }
    
    // Sort by frequency
    startupTabs.sort((a, b) => b.frequency - a.frequency);
    
    return startupTabs;
    
  } catch (error) {
    console.error('Failed to get startup predictions:', error);
    return [];
  }
}

/**
 * Get all learned patterns for UI display
 */
export async function getLearnedPatterns() {
  try {
    const patterns = await storage.getAllPatterns();
    
    return patterns
      .filter(p => p.importance > 0)
      .sort((a, b) => b.importance - a.importance)
      .map(p => ({
        domain: p.domain,
        importance: p.importance,
        visitCount: p.visitCount,
        totalTime: p.totalTimeSpent,
        reopens: p.reopens,
        startupFrequency: p.startupAppearances,
        peakHours: getPeakHours(p.hourDistribution),
        peakDays: getPeakDays(p.dayDistribution)
      }));
    
  } catch (error) {
    console.error('Failed to get learned patterns:', error);
    return [];
  }
}

/**
 * Get peak usage hours
 */
function getPeakHours(hourDistribution) {
  const threshold = Math.max(...hourDistribution) * 0.5;
  return hourDistribution
    .map((count, hour) => ({ hour, count }))
    .filter(h => h.count >= threshold)
    .map(h => h.hour);
}

/**
 * Get peak usage days
 */
function getPeakDays(dayDistribution) {
  const days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
  const threshold = Math.max(...dayDistribution) * 0.5;
  return dayDistribution
    .map((count, day) => ({ day: days[day], count }))
    .filter(d => d.count >= threshold)
    .map(d => d.day);
}
