/**
 * Performance Monitoring and Protection
 * Handles tab suspension and cleanup
 */

import { storage } from './storage.js';

// Performance thresholds
const PERFORMANCE_CONFIG = {
  MAX_TABS_WARNING: 30,
  MAX_TABS_CRITICAL: 50,
  CLEANUP_SIGNALS_DAYS: 30,
  CLEANUP_VAULT_DAYS: 90
};

/**
 * Monitor system performance and take action if needed
 */
export async function monitorPerformance() {
  const openTabs = await chrome.tabs.query({});
  const tabCount = openTabs.length;
  
  if (tabCount >= PERFORMANCE_CONFIG.MAX_TABS_CRITICAL) {
    console.warn(`Critical tab count: ${tabCount}`);
    // Trigger aggressive archiving
    return { status: 'critical', tabCount };
  } else if (tabCount >= PERFORMANCE_CONFIG.MAX_TABS_WARNING) {
    console.warn(`Warning tab count: ${tabCount}`);
    return { status: 'warning', tabCount };
  }
  
  return { status: 'normal', tabCount };
}

/**
 * Cleanup old signals and vault entries
 */
export async function performCleanup() {
  try {
    console.log('Starting cleanup...');
    
    // Clean old signals
    await storage.cleanupOldSignals(PERFORMANCE_CONFIG.CLEANUP_SIGNALS_DAYS);
    
    // Clean old vault entries
    await storage.cleanupOldVaultEntries(PERFORMANCE_CONFIG.CLEANUP_VAULT_DAYS);
    
    console.log('Cleanup completed');
    
  } catch (error) {
    console.error('Cleanup failed:', error);
  }
}

/**
 * Get storage usage stats
 */
export async function getStorageStats() {
  try {
    const tabs = await storage.getAllTabs();
    const vault = await storage.getAllVaultEntries();
    const signals = await storage.getAll('signals');
    
    // Estimate storage size
    const estimate = await navigator.storage?.estimate();
    
    return {
      tabCount: tabs.length,
      vaultCount: vault.length,
      signalCount: signals.length,
      storageUsed: estimate?.usage || 0,
      storageQuota: estimate?.quota || 0,
      usagePercent: estimate ? (estimate.usage / estimate.quota * 100).toFixed(2) : 0
    };
  } catch (error) {
    console.error('Stats collection failed:', error);
    return null;
  }
}

/**
 * Update performance configuration
 */
export function updatePerformanceConfig(config) {
  Object.assign(PERFORMANCE_CONFIG, config);
}
