/**
 * Contextual Resurfacing
 * Suggests relevant vault entries based on current context
 */

import { storage } from './storage.js';

/**
 * Get suggestions based on context
 * @param {Object} context - { type: 'new_tab'|'address_bar'|'domain', query: string, domain: string }
 */
export async function getSuggestions(context) {
  const { type, query, domain } = context;
  
  switch (type) {
    case 'new_tab':
      return getNewTabSuggestions();
      
    case 'address_bar':
      return getAddressBarSuggestions(query);
      
    case 'domain':
      return getDomainSuggestions(domain);
      
    default:
      return [];
  }
}

/**
 * Suggestions for new tab - show recent high-value items
 */
async function getNewTabSuggestions() {
  const vaultEntries = await storage.getAllVaultEntries();
  
  // Sort by combination of score and recency
  const scored = vaultEntries.map(entry => ({
    ...entry,
    relevance: calculateNewTabRelevance(entry)
  }));
  
  scored.sort((a, b) => b.relevance - a.relevance);
  
  return scored.slice(0, 10);
}

/**
 * Suggestions for address bar input - keyword matching
 */
async function getAddressBarSuggestions(query) {
  if (!query || query.length < 2) return [];
  
  const vaultEntries = await storage.getAllVaultEntries();
  const lowerQuery = query.toLowerCase();
  
  // Score entries by relevance to query
  const scored = vaultEntries.map(entry => ({
    ...entry,
    relevance: calculateQueryRelevance(entry, lowerQuery)
  }));
  
  // Filter out low-relevance results
  const filtered = scored.filter(entry => entry.relevance > 0);
  filtered.sort((a, b) => b.relevance - a.relevance);
  
  return filtered.slice(0, 10);
}

/**
 * Suggestions for domain revisit - show other pages from same domain
 */
async function getDomainSuggestions(domain) {
  if (!domain) return [];
  
  const entries = await storage.getVaultEntriesByDomain(domain);
  
  // Sort by score
  entries.sort((a, b) => (b.score || 0) - (a.score || 0));
  
  return entries.slice(0, 5);
}

/**
 * Calculate relevance for new tab context
 * Balance between recency and original score
 */
function calculateNewTabRelevance(entry) {
  const hoursAgo = (Date.now() - entry.archivedAt) / (1000 * 60 * 60);
  const recencyFactor = Math.exp(-hoursAgo / 48); // 48-hour decay
  
  const scoreNormalized = (entry.score || 0) / 100;
  
  return (recencyFactor * 0.6 + scoreNormalized * 0.4);
}

/**
 * Calculate relevance for query matching
 * Simple keyword matching (will be replaced with TF-IDF in search module)
 */
function calculateQueryRelevance(entry, query) {
  let score = 0;
  
  // Check title
  if (entry.title?.toLowerCase().includes(query)) {
    score += 10;
  }
  
  // Check URL
  if (entry.url?.toLowerCase().includes(query)) {
    score += 5;
  }
  
  // Check text extract
  if (entry.textExtract?.toLowerCase().includes(query)) {
    score += 3;
  }
  
  // Check domain
  if (entry.domain?.toLowerCase().includes(query)) {
    score += 7;
  }
  
  // Boost by original score
  score *= (1 + (entry.score || 0) / 100);
  
  return score;
}

/**
 * Detect if a URL is being revisited
 */
export async function checkDomainRevisit(url) {
  try {
    const urlObj = new URL(url);
    const domain = urlObj.hostname;
    
    const suggestions = await getDomainSuggestions(domain);
    
    if (suggestions.length > 0) {
      return {
        isDomainRevisit: true,
        domain,
        suggestions
      };
    }
  } catch (error) {
    console.error('Domain check failed:', error);
  }
  
  return { isDomainRevisit: false };
}
