/**
 * Tab Importance Scoring Engine
 * Calculates relevance scores based on dwell time, revisits, interactions, and recency
 */

import { storage } from './storage.js';

// Scoring weights (tunable)
const WEIGHTS = {
  DWELL: 0.4,
  REVISIT: 0.3,
  INTERACTION: 0.2,
  RECENCY: 0.1
};

// Manual pin bonus
const MANUAL_PIN_BONUS = 50;

// Thresholds
export const THRESHOLDS = {
  ARCHIVE: 10,  // Below this, eligible for archiving
  SUSPEND: 5    // Below this, suspend to free memory
};

// Normalization constants
const MAX_DWELL_TIME = 600000; // 10 minutes
const MAX_REVISITS = 20;
const MAX_INTERACTIONS = 10;
const RECENCY_DECAY_HOURS = 24;

/**
 * Calculate normalized dwell time score (0-1)
 */
function calculateDwellScore(dwellTime) {
  if (!dwellTime) return 0;
  return Math.min(dwellTime / MAX_DWELL_TIME, 1);
}

/**
 * Calculate normalized revisit score (0-1)
 */
function calculateRevisitScore(activationCount) {
  if (!activationCount) return 0;
  return Math.min(activationCount / MAX_REVISITS, 1);
}

/**
 * Calculate normalized interaction score (0-1)
 */
function calculateInteractionScore(signals) {
  if (!signals) return 0;
  
  const interactions = 
    (signals.copyEvents || 0) +
    (signals.pasteEvents || 0) +
    (signals.clickEvents || 0) * 0.1; // Clicks weighted less
  
  return Math.min(interactions / MAX_INTERACTIONS, 1);
}

/**
 * Calculate recency factor (0-1)
 * More recent = higher score, with exponential decay
 */
function calculateRecencyScore(lastActivated) {
  if (!lastActivated) return 0;
  
  const hoursAgo = (Date.now() - lastActivated) / (1000 * 60 * 60);
  const decayFactor = Math.exp(-hoursAgo / RECENCY_DECAY_HOURS);
  
  return decayFactor;
}

/**
 * Calculate composite importance score for a tab
 */
export async function calculateScore(tabId) {
  const tab = await storage.getTab(tabId);
  if (!tab) return 0;
  
  // Get aggregated signals for this tab
  const signals = await getAggregatedSignals(tabId);
  
  // Calculate component scores
  const dwellScore = calculateDwellScore(signals.totalDwellTime);
  const revisitScore = calculateRevisitScore(tab.activationCount || 0);
  const interactionScore = calculateInteractionScore(signals);
  const recencyScore = calculateRecencyScore(tab.lastActivated);
  
  // Weighted composite score
  let score = 
    WEIGHTS.DWELL * dwellScore +
    WEIGHTS.REVISIT * revisitScore +
    WEIGHTS.INTERACTION * interactionScore +
    WEIGHTS.RECENCY * recencyScore;
  
  // Scale to 0-100
  score *= 100;
  
  // Add manual pin bonus
  if (tab.pinned) {
    score += MANUAL_PIN_BONUS;
  }
  
  return Math.round(score * 10) / 10; // Round to 1 decimal
}

/**
 * Aggregate signals for a tab
 */
async function getAggregatedSignals(tabId) {
  const signals = await storage.getSignalsForTab(tabId);
  
  const aggregated = {
    totalDwellTime: 0,
    copyEvents: 0,
    pasteEvents: 0,
    clickEvents: 0,
    maxScrollDepth: 0
  };
  
  for (const signal of signals) {
    aggregated.totalDwellTime += signal.dwellTime || 0;
    aggregated.copyEvents += signal.copyEvents || 0;
    aggregated.pasteEvents += signal.pasteEvents || 0;
    aggregated.clickEvents += signal.clickEvents || 0;
    aggregated.maxScrollDepth = Math.max(
      aggregated.maxScrollDepth,
      signal.scrollDepth || 0
    );
  }
  
  return aggregated;
}

/**
 * Update score for a specific tab
 */
export async function updateTabScore(tabId) {
  const score = await calculateScore(tabId);
  await storage.updateTab(tabId, { 
    score,
    scoreUpdatedAt: Date.now()
  });
  
  return score;
}

/**
 * Get current score for a tab
 */
export async function getTabScore(tabId) {
  const tab = await storage.getTab(tabId);
  return tab?.score || 0;
}

/**
 * Get tabs below archive threshold
 */
export async function getArchivableTabs() {
  // Get threshold from settings
  const archiveThreshold = await storage.getSetting('archiveThreshold', THRESHOLDS.ARCHIVE);
  
  const tabs = await storage.getAllTabs();
  
  return tabs.filter(tab => 
    !tab.pinned && 
    (tab.score || 0) < archiveThreshold
  );
}

/**
 * Get tabs below suspend threshold
 */
export async function getSuspendableTabs() {
  const tabs = await storage.getAllTabs();
  
  return tabs.filter(tab =>
    !tab.pinned &&
    (tab.score || 0) < THRESHOLDS.SUSPEND
  );
}

/**
 * Update scoring weights (for tuning)
 */
export function updateWeights(newWeights) {
  Object.assign(WEIGHTS, newWeights);
}

/**
 * Update thresholds (for tuning)
 */
export function updateThresholds(newThresholds) {
  Object.assign(THRESHOLDS, newThresholds);
}
