/**
 * Mnemo Background Service Worker
 * Manages tab lifecycle, scoring, archiving, and message routing
 */

import { storage } from './storage.js';
import { updateTabScore, getTabScore } from './scoring.js';
import { checkAndArchiveTabs, manuallyArchiveTab as archiveTab } from './archiver.js';
import { getSuggestions } from './resurfacing.js';
import { getStorageStats } from './performance.js';
import { getPersonalizationScore, trackVisit, trackRestore, trackStartup, getPredictedStartupTabs, getLearnedPatterns } from './patterns.js';
import { searchWithTFIDF } from './search.js';

// Constants
const SCORING_INTERVAL = 30000; // 30 seconds
const ARCHIVE_INTERVAL = 120000; // 2 minutes

// Track active tab signals
const tabSignals = new Map();

/**
 * Initialize extension on install
 */
chrome.runtime.onInstalled.addListener(async () => {
  console.log('Mnemo installed');
  
  // Initialize storage
  await storage.initialize();
  
  // Set up periodic tasks
  chrome.alarms.create('updateScores', { periodInMinutes: 0.5 }); // 30s
  chrome.alarms.create('archiveTabs', { periodInMinutes: 2 });
  
  // Enable side panel
  chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });
});

/**
 * Initialize storage when service worker starts
 */
(async () => {
  console.log('Mnemo service worker starting...');
  try {
    await storage.initialize();
    console.log('Mnemo initialized');
    
    // Set up context menus
    setupContextMenus();
  } catch (error) {
    console.error('Failed to initialize storage:', error);
  }
})();

/**
 * Browser startup - Smart startup tabs
 */
chrome.runtime.onStartup.addListener(async () => {
  const enableSmartStartup = await storage.getSetting('enableSmartStartup', false);
  const enableML = await storage.getSetting('enableMLPersonalization', true);
  
  if (enableSmartStartup && enableML) {
    // Wait for browser to initialize
    setTimeout(async () => {
      try {
        const context = {
          dayOfWeek: new Date().getDay(),
          hour: new Date().getHours()
        };
        
        const predictions = await getPredictedStartupTabs(context);
        console.log('Smart startup predictions:', predictions);
        
        // Auto-restore predicted tabs
        for (const prediction of predictions) {
          const exists = await chrome.tabs.query({url: `*://${prediction.domain}/*`});
          if (!exists.length) {
            await chrome.tabs.create({
              url: `https://${prediction.domain}`,
              active: false
            });
            console.log(`Smart startup: Opened ${prediction.domain} (${prediction.confidence}% confidence)`);
          }
        }
      } catch (error) {
        console.error('Smart startup failed:', error);
      }
    }, 5000);  // Wait 5s for initial tabs
  }
  
  // Track startup behavior for learning (after 5 minutes)
  if (enableML) {
    setTimeout(async () => {
      try {
        const allTabs = await chrome.tabs.query({});
        const domains = [];
        
        for (const tab of allTabs) {
          try {
            const domain = new URL(tab.url).hostname;
            if (domain) domains.push(domain);
          } catch (e) {
            // Invalid URL
          }
        }
        
        if (domains.length > 0) {
          await trackStartup(domains);
          console.log('Tracked startup tabs for learning:', domains);
        }
      } catch (error) {
        console.error('Startup tracking failed:', error);
      }
    }, 300000);  // 5 minutes
  }
});

/**
 * Set up right-click context menus
 */
function setupContextMenus() {
  chrome.contextMenus.removeAll(() => {
    chrome.contextMenus.create({
      id: 'archive-tab',
      title: 'Archive this tab',
      contexts: ['page']
    });
    
    chrome.contextMenus.create({
      id: 'archive-domain-tabs',
      title: 'Archive all tabs from this domain',
      contexts: ['page']
    });
  });
}

/**
 * Handle context menu clicks
 */
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === 'archive-tab') {
    await manuallyArchiveTab(tab.id);
  } else if (info.menuItemId === 'archive-domain-tabs') {
    await archiveTabsFromDomain(tab.url);
  }
});

/**
 * Handle keyboard shortcuts
 */
chrome.commands.onCommand.addListener(async (command) => {
  switch (command) {
    case 'archive-current-tab':
      const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (activeTab) {
        await manuallyArchiveTab(activeTab.id);
      }
      break;
      
    case 'open-vault':
      await chrome.sidePanel.open({ windowId: (await chrome.windows.getCurrent()).id });
      break;
      
    case 'restore-last':
      await restoreLastArchivedTab();
      break;
  }
});

/**
 * Handle tab activation (user switches tabs)
 */
chrome.tabs.onActivated.addListener(async ({ tabId, windowId }) => {
  try {
    const tab = await chrome.tabs.get(tabId);
    if (!tab || !tab.url) return;
    
    // Track activation for ML learning
    const enableML = await storage.getSetting('enableMLPersonalization', true);
    if (enableML) {
      try {
        const domain = new URL(tab.url).hostname;
        const lastActivation = Date.now();
        
        // Store for duration calculation
        await storage.updateTab(tabId, { 
          lastActivated: lastActivation,
          _mlDomain: domain
        });
      } catch (e) {
        // Invalid URL
      }
    }
    
    await updateTabScore(tabId);
  } catch (error) {
    console.warn(`Failed to handle tab activation:`, error);
  }
});

/**
 * Handle tab updates (URL changes, loading states)
 */
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete') {
    // Store initial tab metadata
    await storage.updateTab(tabId, {
      url: tab.url,
      title: tab.title,
      createdAt: Date.now(),
      lastUpdated: Date.now()
    });
    
    // Check for duplicates
    await checkAndCloseDuplicates(tabId, tab.url);
  }
});

/**
 * Handle tab removal
 */
chrome.tabs.onRemoved.addListener(async (tabId, removeInfo) => {
  // Track visit duration for ML before cleanup
  const enableML = await storage.getSetting('enableMLPersonalization', true);
  if (enableML) {
    try {
      const tabData = await storage.getTab(tabId);
      if (tabData && tabData._mlDomain && tabData.lastActivated) {
        const duration = Date.now() - tabData.lastActivated;
        if (duration > 5000) { // At least 5 seconds
          await trackVisit(tabData._mlDomain, duration);
        }
      }
    } catch (e) {
      // Ignore errors
    }
  }
  
  await storage.deleteTab(tabId);
  tabSignals.delete(tabId);
});

/**
 * Handle messages from content scripts and UI
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  (async () => {
    try {
      switch (message.type) {
        case 'SIGNAL_BATCH':
          await handleSignalBatch(message.data, sender.tab?.id);
          sendResponse({ success: true });
          break;
          
        case 'GET_TAB_SCORE':
          const score = await getTabScore(message.tabId);
          sendResponse({ score });
          break;
          
        case 'GET_SUGGESTIONS':
          const suggestions = await getSuggestions(message.context);
          sendResponse({ suggestions });
          break;
          
        case 'RESTORE_FROM_VAULT':
          await restoreFromVault(message.vaultId);
          sendResponse({ success: true });
          break;
          
        case 'SEARCH_VAULT':
          const results = await searchVault(message.query);
          sendResponse({ results });
          break;
          
        case 'GET_VAULT_ENTRIES':
          const entries = await storage.getAllVaultEntries();
          sendResponse({ entries });
          break;
          
        case 'DELETE_VAULT_ENTRY':
          await storage.deleteVaultEntry(message.vaultId);
          sendResponse({ success: true });
          break;
          
        case 'GET_SETTINGS':
          const settings = {
            archiveThreshold: await storage.getSetting('archiveThreshold', 10),
            maxActiveTabs: await storage.getSetting('maxActiveTabs', 20),
            minInactiveMinutes: await storage.getSetting('minInactiveMinutes', 30),
            autoArchive: await storage.getSetting('autoArchive', true),
            enableDuplicateDetection: await storage.getSetting('enableDuplicateDetection', true),
            captureScreenshots: await storage.getSetting('captureScreenshots', true),
            vaultRetention: await storage.getSetting('vaultRetention', 90)
          };
          sendResponse({ settings });
          break;
          
        case 'SAVE_SETTINGS':
          for (const [key, value] of Object.entries(message.settings)) {
            await storage.setSetting(key, value);
          }
          sendResponse({ success: true });
          break;
          
        case 'GET_STORAGE_STATS':
          const stats = await getStorageStats();
          sendResponse({ stats });
          break;
          
        case 'CLEAR_VAULT':
          const vaultEntries = await storage.getAllVaultEntries();
          for (const entry of vaultEntries) {
            await storage.deleteVaultEntry(entry.id);
          }
          sendResponse({ success: true, count: vaultEntries.length });
          break;
          
        case 'CLEAR_OLD_ENTRIES':
          const retentionDays = await storage.getSetting('vaultRetention', 90);
          if (retentionDays > 0) {
            await storage.cleanupOldVaultEntries(retentionDays);
          }
          sendResponse({ success: true });
          break;
          
        case 'CLEAR_HISTORY':
          await storage.cleanupOldSignals(0); // Delete all signals
          sendResponse({ success: true });
          break;
          
        case 'GET_DOMAIN_RULES':
          const rules = await storage.getSetting('domainRules', []);
          sendResponse({ rules });
          break;
          
        case 'ADD_DOMAIN_RULE':
          const currentRules = await storage.getSetting('domainRules', []);
          // Remove existing rule for same domain
          const filteredRules = currentRules.filter(r => r.domain !== message.rule.domain);
          filteredRules.push(message.rule);
          await storage.setSetting('domainRules', filteredRules);
          sendResponse({ success: true });
          break;
          
        case 'DELETE_DOMAIN_RULE':
          const existingRules = await storage.getSetting('domainRules', []);
          const updatedRules = existingRules.filter(r => r.domain !== message.domain);
          await storage.setSetting('domainRules', updatedRules);
          sendResponse({ success: true });
          break;
          
        case 'GET_LEARNED_PATTERNS':
          try {
            const patterns = await getLearnedPatterns();
            sendResponse({ patterns });
          } catch (error) {
            console.error('Failed to get learned patterns:', error);
            sendResponse({ patterns: [] });
          }
          break;
          
        case 'PURGE_ALL_DATA':
          await storage.purgeAllData();
          sendResponse({ success: true });
          break;
          
        default:
          sendResponse({ error: 'Unknown message type' });
      }
    } catch (error) {
      console.error('Message handler error:', error);
      sendResponse({ error: error.message });
    }
  })();
  
  return true; // Keep message channel open for async response
});

/**
 * Handle alarm events for periodic tasks
 */
if (chrome.alarms) {
  chrome.alarms.onAlarm.addListener(async (alarm) => {
    try {
      switch (alarm.name) {
        case 'updateScores':
          await updateAllTabScores();
          break;
          
        case 'archiveTabs':
          await checkAndArchiveTabs();
          break;
      }
    } catch (error) {
      console.error('Alarm handler error:', error);
    }
  });
} else {
  console.warn('chrome.alarms API not available');
}

/**
 * Process signal batch from content script
 */
async function handleSignalBatch(signals, tabId) {
  if (!tabId) return;
  
  // Ensure storage is ready
  if (!storage.db) {
    console.warn('Storage not ready, skipping signal batch');
    return;
  }
  
  try {
    // Store signals
    await storage.addSignals(tabId, signals);
    
    // Update local cache for quick access
    tabSignals.set(tabId, {
      ...tabSignals.get(tabId),
      ...signals,
      lastUpdate: Date.now()
    });
    
    // Recalculate score with new signals
    await updateTabScore(tabId);
  } catch (error) {
    console.error('Failed to handle signal batch:', error);
  }
}

/**
 * Update scores for all tracked tabs
 */
async function updateAllTabScores() {
  const tabs = await chrome.tabs.query({});
  
  for (const tab of tabs) {
    await updateTabScore(tab.id);
  }
}

/**
 * Restore a tab from vault
 */
async function restoreFromVault(vaultId) {
  const entry = await storage.getVaultEntry(vaultId);
  if (!entry) return;
  
  // Create new tab
  const tab = await chrome.tabs.create({
    url: entry.url,
    active: true
  });
  
  // Track restore for ML learning
  const enableML = await storage.getSetting('enableMLPersonalization', true);
  if (enableML) {
    try {
      const domain = new URL(entry.url).hostname;
      await trackRestore(domain);
      console.log(`ML: Tracked restore of ${domain}`);
    } catch (e) {
      // Ignore
    }
  }
  
  // Remove from vault
  await storage.deleteVaultEntry(vaultId);
  
  console.log('Restored tab from vault:', entry.url);
  
  // Wait for tab to load
  await new Promise((resolve) => {
    const listener = (tabId, changeInfo) => {
      if (tabId === tab.id && changeInfo.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(listener);
        resolve();
      }
    };
    chrome.tabs.onUpdated.addListener(listener);
  });
  
  // Restore scroll position
  if (entry.scrollPosition) {
    await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: (scrollPos) => {
        window.scrollTo(0, scrollPos);
      },
      args: [entry.scrollPosition]
    });
  }
  
  // Boost score for restored tab
  await storage.updateTab(tab.id, {
    restoredFrom: vaultId,
    restoredAt: Date.now()
  });
  
  await updateTabScore(tab.id);
}

/**
 * Search vault entries with TF-IDF ranking
 */
async function searchVault(query) {
  try {
    return await searchWithTFIDF(query);
  } catch (error) {
    console.error('TF-IDF search failed, falling back to basic search:', error);
    
    // Fallback to basic search
    const vaultEntries = await storage.getAllVaultEntries();
    const lowerQuery = query.toLowerCase();
    return vaultEntries.filter(entry => 
      entry.title?.toLowerCase().includes(lowerQuery) ||
      entry.textExtract?.toLowerCase().includes(lowerQuery) ||
      entry.url?.toLowerCase().includes(lowerQuery)
    ).slice(0, 20);
  }
}

/**
 * Manually archive a specific tab (called from context menu or keyboard shortcut)
 */
async function manuallyArchiveTab(tabId) {
  try {
    await archiveTab(tabId);
    console.log(`Manually archived tab ${tabId}`);
  } catch (error) {
    console.error('Manual archive failed:', error);
  }
}

/**
 * Archive all tabs from a specific domain
 */
async function archiveTabsFromDomain(url) {
  try {
    const domain = new URL(url).hostname;
    const tabs = await chrome.tabs.query({});
    let archivedCount = 0;
    
    for (const tab of tabs) {
      try {
        const tabDomain = new URL(tab.url).hostname;
        if (tabDomain === domain && !tab.active && !tab.pinned) {
          await manuallyArchiveTab(tab.id);
          archivedCount++;
        }
      } catch (e) {
        // Skip invalid URLs
      }
    }
    
    console.log(`Archived ${archivedCount} tabs from ${domain}`);
  } catch (error) {
    console.error('Archive domain tabs failed:', error);
  }
}

/**
 * Restore the most recently archived tab
 */
async function restoreLastArchivedTab() {
  try {
    const vaultEntries = await storage.getAllVaultEntries();
    if (vaultEntries.length === 0) {
      console.log('No archived tabs to restore');
      return;
    }
    
    // Sort by archivedAt descending
    vaultEntries.sort((a, b) => b.archivedAt - a.archivedAt);
    const lastEntry = vaultEntries[0];
    
    // Restore it
    await restoreFromVault(lastEntry.id);
    console.log('Restored last archived tab');
  } catch (error) {
    console.error('Restore last failed:', error);
  }
}

/**
 * Check for and close duplicate tabs
 */
async function checkAndCloseDuplicates(newTabId, newUrl) {
  try {
    // Check if duplicate detection is enabled
    const enableDuplicateDetection = await storage.getSetting('enableDuplicateDetection', true);
    if (!enableDuplicateDetection) return;
    
    const tabs = await chrome.tabs.query({});
    const duplicates = [];
    
    for (const tab of tabs) {
      if (tab.id !== newTabId && tab.url === newUrl && !tab.pinned) {
        duplicates.push(tab);
      }
    }
    
    if (duplicates.length > 0) {
      console.log(`Found ${duplicates.length} duplicate(s) of ${newUrl}`);
      
      // Close older duplicates (keep the new one)
      for (const duplicate of duplicates) {
        try {
          await chrome.tabs.remove(duplicate.id);
          console.log(`Closed duplicate tab ${duplicate.id}`);
        } catch (error) {
          console.warn(`Failed to close duplicate tab ${duplicate.id}:`, error);
        }
      }
    }
  } catch (error) {
    console.error('Duplicate detection failed:', error);
  }
}

console.log('Mnemo service worker loaded');
