/**
 * IndexedDB Storage Layer for Mnemo
 * Manages persistent storage of tabs, vault, signals, and settings
 */

const DB_NAME = 'MnemoDB';
const DB_VERSION = 1;

// Store names
const STORES = {
  TABS: 'tabs',
  VAULT: 'vault',
  SIGNALS: 'signals',
  SETTINGS: 'settings',
  INDEX: 'searchIndex',
  PATTERNS: 'patterns',  // ML learning data
  STARTUP_SESSIONS: 'startupSessions'  // Startup tracking
};

class Storage {
  constructor() {
    this.db = null;
  }

  /**
   * Initialize IndexedDB
   */
  async initialize() {
    return new Promise((resolve, reject) => {
      const request = indexedDB.open(DB_NAME, DB_VERSION);

      request.onerror = () => reject(request.error);
      request.onsuccess = () => {
        this.db = request.result;
        resolve();
      };

      request.onupgradeneeded = (event) => {
        const db = event.target.result;

        // Tabs store - active tab metadata and scores
        if (!db.objectStoreNames.contains(STORES.TABS)) {
          const tabStore = db.createObjectStore(STORES.TABS, { keyPath: 'tabId' });
          tabStore.createIndex('score', 'score', { unique: false });
          tabStore.createIndex('lastActivated', 'lastActivated', { unique: false });
          tabStore.createIndex('url', 'url', { unique: false });
        }

        // Vault store - archived tabs
        if (!db.objectStoreNames.contains(STORES.VAULT)) {
          const vaultStore = db.createObjectStore(STORES.VAULT, { keyPath: 'id' });
          vaultStore.createIndex('archivedAt', 'archivedAt', { unique: false });
          vaultStore.createIndex('score', 'score', { unique: false });
          vaultStore.createIndex('url', 'url', { unique: false });
          vaultStore.createIndex('domain', 'domain', { unique: false });
        }

        // Signals store - interaction signals
        if (!db.objectStoreNames.contains(STORES.SIGNALS)) {
          const signalStore = db.createObjectStore(STORES.SIGNALS, { 
            keyPath: 'id',
            autoIncrement: true 
          });
          signalStore.createIndex('tabId', 'tabId', { unique: false });
          signalStore.createIndex('timestamp', 'timestamp', { unique: false });
        }

        // Search index - TF-IDF search data
        if (!db.objectStoreNames.contains(STORES.INDEX)) {
          db.createObjectStore(STORES.INDEX, { keyPath: 'vaultId' });
        }
        
        // Patterns store - ML learning data
        if (!db.objectStoreNames.contains(STORES.PATTERNS)) {
          const patternStore = db.createObjectStore(STORES.PATTERNS, { keyPath: 'domain' });
          patternStore.createIndex('importance', 'importance', { unique: false });
        }
        
        // Startup sessions - Track browser startup behavior
        if (!db.objectStoreNames.contains(STORES.STARTUP_SESSIONS)) {
          db.createObjectStore(STORES.STARTUP_SESSIONS, { autoIncrement: true });
        }

        // Settings store
        if (!db.objectStoreNames.contains(STORES.SETTINGS)) {
          db.createObjectStore(STORES.SETTINGS, { keyPath: 'key' });
        }
      };
    });
  }

  /**
   * Generic get operation
   */
  async get(storeName, key) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction([storeName], 'readonly');
      const store = transaction.objectStore(storeName);
      const request = store.get(key);

      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }

  /**
   * Generic put operation
   */
  async put(storeName, data) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction([storeName], 'readwrite');
      const store = transaction.objectStore(storeName);
      const request = store.put(data);

      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }

  /**
   * Generic delete operation
   */
  async delete(storeName, key) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction([storeName], 'readwrite');
      const store = transaction.objectStore(storeName);
      const request = store.delete(key);

      request.onsuccess = () => resolve();
      request.onerror = () => reject(request.error);
    });
  }

  /**
   * Get all entries from a store
   */
  async getAll(storeName) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction([storeName], 'readonly');
      const store = transaction.objectStore(storeName);
      const request = store.getAll();

      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }

  /**
   * Query by index
   */
  async queryByIndex(storeName, indexName, value) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction([storeName], 'readonly');
      const store = transaction.objectStore(storeName);
      const index = store.index(indexName);
      const request = index.getAll(value);

      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }

  // Tab-specific methods
  async getTab(tabId) {
    return this.get(STORES.TABS, tabId);
  }

  async updateTab(tabId, data) {
    const existing = await this.getTab(tabId) || { tabId };
    return this.put(STORES.TABS, { ...existing, ...data });
  }

  async deleteTab(tabId) {
    return this.delete(STORES.TABS, tabId);
  }

  async getAllTabs() {
    return this.getAll(STORES.TABS);
  }

  async getTabsByScore(minScore) {
    const tabs = await this.getAllTabs();
    return tabs.filter(tab => (tab.score || 0) >= minScore);
  }

  // Vault-specific methods
  async getVaultEntry(id) {
    return this.get(STORES.VAULT, id);
  }

  async addVaultEntry(entry) {
    const id = `vault_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    return this.put(STORES.VAULT, { ...entry, id });
  }

  async deleteVaultEntry(id) {
    return this.delete(STORES.VAULT, id);
  }

  async getAllVaultEntries() {
    return this.getAll(STORES.VAULT);
  }

  async getVaultEntriesByDomain(domain) {
    return this.queryByIndex(STORES.VAULT, 'domain', domain);
  }

  // Signal-specific methods
  async addSignals(tabId, signals) {
    const signalEntry = {
      tabId,
      timestamp: Date.now(),
      ...signals
    };
    return this.put(STORES.SIGNALS, signalEntry);
  }

  async getSignalsForTab(tabId) {
    return this.queryByIndex(STORES.SIGNALS, 'tabId', tabId);
  }

  async deleteSignalsForTab(tabId) {
    const signals = await this.getSignalsForTab(tabId);
    for (const signal of signals) {
      await this.delete(STORES.SIGNALS, signal.id);
    }
  }

  // Settings methods
  async getSetting(key, defaultValue = null) {
    const result = await this.get(STORES.SETTINGS, key);
    return result ? result.value : defaultValue;
  }

  async setSetting(key, value) {
    return this.put(STORES.SETTINGS, { key, value });
  }

  // Cleanup methods
  async cleanupOldSignals(days) {
    const cutoff = Date.now() - (days * 24 * 60 * 60 * 1000);
    const allSignals = await this.getAll(STORES.SIGNALS);
    
    for (const signal of allSignals) {
      if (signal.timestamp < cutoff) {
        await this.delete(STORES.SIGNALS, signal.id);
      }
    }
  }
  
  /**
   * Search index operations (TF-IDF)
   */
  async updateSearchIndex(vaultId, terms) {
    return this.put(STORES.INDEX, { vaultId, terms });
  }
  
  async getSearchIndex(vaultId) {
    return this.get(STORES.INDEX, vaultId);
  }
  
  async getAllSearchIndexes() {
    return this.getAll(STORES.INDEX);
  }
  
  async deleteSearchIndex(vaultId) {
    return this.delete(STORES.INDEX, vaultId);
  }
  
  /**
   * Pattern operations (ML learning)
   */
  async savePattern(pattern) {
    return this.put(STORES.PATTERNS, pattern);
  }
  
  async getPattern(domain) {
    return this.get(STORES.PATTERNS, domain);
  }
  
  async getAllPatterns() {
    return this.getAll(STORES.PATTERNS);
  }
  
  async deletePattern(domain) {
    return this.delete(STORES.PATTERNS, domain);
  }
  
  /**
   * Startup session operations
   */
  async saveStartupSessions(sessions) {
    // Clear and save all
    const transaction = this.db.transaction([STORES.STARTUP_SESSIONS], 'readwrite');
    const store = transaction.objectStore(STORES.STARTUP_SESSIONS);
    await new Promise((resolve, reject) => {
      const clearReq = store.clear();
      clearReq.onsuccess = () => resolve();
      clearReq.onerror = () => reject(clearReq.error);
    });
    
    for (const session of sessions) {
      await this.put(STORES.STARTUP_SESSIONS, session);
    }
  }
  
  async getStartupSessions() {
    return this.getAll(STORES.STARTUP_SESSIONS);
  }

  async cleanupOldVaultEntries(daysToKeep = 90) {
    const cutoff = Date.now() - (daysToKeep * 24 * 60 * 60 * 1000);
    const allEntries = await this.getAllVaultEntries();
    
    for (const entry of allEntries) {
      if (entry.archivedAt < cutoff) {
        await this.deleteVaultEntry(entry.id);
      }
    }
  }

  /**
   * Purge all data (for privacy/reset)
   */
  async purgeAllData() {
    const stores = [STORES.TABS, STORES.VAULT, STORES.SIGNALS, STORES.INDEX];
    
    for (const storeName of stores) {
      const transaction = this.db.transaction([storeName], 'readwrite');
      const store = transaction.objectStore(storeName);
      await new Promise((resolve, reject) => {
        const request = store.clear();
        request.onsuccess = () => resolve();
        request.onerror = () => reject(request.error);
      });
    }
  }
}

// Export singleton instance
export const storage = new Storage();
