/**
 * Content Script - Signal Capture
 * Passively collects interaction signals and sends to background worker
 */

// Signal tracking state
const signals = {
  dwellTime: 0,
  scrollDepth: 0,
  copyEvents: 0,
  pasteEvents: 0,
  clickEvents: 0,
  focusTime: 0
};

// Timing state
let pageLoadTime = Date.now();
let lastVisibleTime = Date.now();
let isVisible = !document.hidden;
let isFocused = document.hasFocus();

// Batch sending interval
const BATCH_INTERVAL = 30000; // 30 seconds

/**
 * Track dwell time (time spent with page visible and focused)
 */
function startDwellTracking() {
  setInterval(() => {
    if (isVisible && isFocused) {
      signals.dwellTime += 1000; // Add 1 second
    }
  }, 1000);
}

/**
 * Track scroll depth
 */
function trackScrollDepth() {
  const calculateScrollDepth = () => {
    const windowHeight = window.innerHeight;
    const documentHeight = document.documentElement.scrollHeight;
    const scrollTop = window.scrollY || document.documentElement.scrollTop;
    
    const scrollPercent = (scrollTop + windowHeight) / documentHeight;
    signals.scrollDepth = Math.max(signals.scrollDepth, Math.min(scrollPercent, 1));
  };
  
  // Track initial position
  calculateScrollDepth();
  
  // Track on scroll with throttling
  let scrollTimeout;
  window.addEventListener('scroll', () => {
    clearTimeout(scrollTimeout);
    scrollTimeout = setTimeout(calculateScrollDepth, 200);
  }, { passive: true });
}

/**
 * Track copy/paste events
 */
function trackClipboardEvents() {
  document.addEventListener('copy', () => {
    signals.copyEvents++;
  });
  
  document.addEventListener('paste', () => {
    signals.pasteEvents++;
  });
}

/**
 * Track significant clicks
 */
function trackClicks() {
  document.addEventListener('click', (e) => {
    // Only count clicks on links and buttons
    const target = e.target.closest('a, button, [role="button"]');
    if (target) {
      signals.clickEvents++;
    }
  }, { passive: true });
}

/**
 * Track visibility changes
 */
function trackVisibility() {
  document.addEventListener('visibilitychange', () => {
    isVisible = !document.hidden;
    
    if (isVisible) {
      lastVisibleTime = Date.now();
    }
  });
}

/**
 * Track focus changes
 */
function trackFocus() {
  window.addEventListener('focus', () => {
    isFocused = true;
  });
  
  window.addEventListener('blur', () => {
    isFocused = false;
  });
}

/**
 * Extract page metadata and text
 */
function extractPageData() {
  const title = document.title;
  const url = window.location.href;
  
  // Extract meta description
  const metaDesc = document.querySelector('meta[name="description"]');
  const description = metaDesc ? metaDesc.content : '';
  
  // Extract first ~2KB of visible text
  const bodyText = document.body.innerText || '';
  const textExtract = bodyText.substring(0, 2048);
  
  // Get current scroll position
  const scrollPosition = window.scrollY || document.documentElement.scrollTop;
  
  return {
    title,
    url,
    description,
    textExtract,
    scrollPosition
  };
}

/**
 * Send signal batch to background worker
 */
async function sendSignalBatch() {
  try {
    const pageData = extractPageData();
    
    const batch = {
      ...signals,
      ...pageData,
      timestamp: Date.now(),
      timeOnPage: Date.now() - pageLoadTime
    };
    
    // Check if extension context is still valid
    if (!chrome.runtime?.id) {
      console.log('Extension context invalidated, skipping signal send');
      return;
    }
    
    await chrome.runtime.sendMessage({
      type: 'SIGNAL_BATCH',
      data: batch
    });
    
    // Reset counters (keep scroll depth and dwell time cumulative)
    signals.copyEvents = 0;
    signals.pasteEvents = 0;
    signals.clickEvents = 0;
    
  } catch (error) {
    // Extension was reloaded - this is expected, don't spam console
    if (error.message?.includes('Extension context invalidated')) {
      console.log('Mnemo: Extension reloaded');
    } else {
      console.error('Failed to send signal batch:', error);
    }
  }
}

/**
 * Initialize signal capture
 */
function initialize() {
  // Start tracking
  startDwellTracking();
  trackScrollDepth();
  trackClipboardEvents();
  trackClicks();
  trackVisibility();
  trackFocus();
  
  // Send initial batch after 5 seconds
  setTimeout(sendSignalBatch, 5000);
  
  // Send periodic batches
  setInterval(sendSignalBatch, BATCH_INTERVAL);
  
  // Send final batch on page unload
  window.addEventListener('beforeunload', () => {
    sendSignalBatch();
  });
  
  console.log('Mnemo capture initialized');
}

// Start capture when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initialize);
} else {
  initialize();
}
