let currentTab = 'vault';
let vaultEntries = [];
let searchQuery = '';
let sortBy = 'recent';

/**
 * Detect current browser
 */
function detectBrowser() {
  const userAgent = navigator.userAgent.toLowerCase();
  
  if (userAgent.includes('edg/')) return 'edge';
  if (userAgent.includes('brave')) return 'brave';
  if (userAgent.includes('opr/') || userAgent.includes('opera')) return 'opera';
  if (userAgent.includes('firefox')) return 'firefox';
  if (userAgent.includes('chrome')) return 'chrome';
  
  return 'chrome'; // Default fallback
}

/**
 * Get browser logo SVG based on detected browser
 */
function getBrowserLogoSVG() {
  const browser = detectBrowser();
  
  switch (browser) {
    case 'brave':
      return `
        <svg width="64" height="64" viewBox="0 0 436.49 511.97">
          <defs>
            <linearGradient id="brave-gradient" x1="-18.79" y1="359.73" x2="194.32" y2="359.73" gradientTransform="matrix(2.05, 0, 0, -2.05, 38.49, 992.77)" gradientUnits="userSpaceOnUse">
              <stop offset="0" stop-color="#f1562b"/>
              <stop offset="0.3" stop-color="#f1542b"/>
              <stop offset="0.5" stop-color="#ef4029"/>
              <stop offset="1" stop-color="#df3c26"/>
            </linearGradient>
          </defs>
          <path fill="url(#brave-gradient)" d="M436.49,165.63,420.7,122.75l11-24.6A8.47,8.47,0,0,0,430,88.78L400.11,58.6a48.16,48.16,0,0,0-50.23-11.66l-8.19,2.89L296.09.43,218.25,0,140.4.61,94.85,50.41l-8.11-2.87A48.33,48.33,0,0,0,36.19,59.3L5.62,90.05a6.73,6.73,0,0,0-1.36,7.47l11.47,25.56L0,165.92,56.47,380.64a89.7,89.7,0,0,0,34.7,50.23l111.68,75.69a24.73,24.73,0,0,0,30.89,0l111.62-75.8A88.86,88.86,0,0,0,380,380.53l46.07-176.14Z"/>
          <path fill="#FFF" d="M231,317.33a65.61,65.61,0,0,0-9.11-3.3h-5.49a66.08,66.08,0,0,0-9.11,3.3l-13.81,5.74-15.6,7.18-25.4,13.24a4.84,4.84,0,0,0-.62,9l22.06,15.49q7,5,13.55,10.76l6.21,5.35,13,11.37,5.89,5.2a10.15,10.15,0,0,0,12.95,0l25.39-22.18,13.6-10.77,22.06-15.79a4.8,4.8,0,0,0-.68-8.93l-25.36-12.8L244.84,323Z"/>
        </svg>
      `;
    
    case 'arc':
      return `
        <svg width="64" height="64" viewBox="0 0 512 512" fill-rule="evenodd" clip-rule="evenodd" stroke-linejoin="round" stroke-miterlimit="2">
          <path fill="#fff" d="M348.708 140.086l-25.184-51.845c-14.53-29.879-44.983-48.922-78.21-48.922-33.754 0-64.585 19.65-78.835 50.236l-34.722 74.533c-25.906-15.186-57.903-15.908-84.466-1.921C5.756 183.854-8.805 235.239 8.547 275.132c9.916 22.951 24.38 44.26 42.11 63.157l-9.128 19.634c-11.951 25.594-10.671 55.473 3.398 79.967a85.722 85.722 0 0031.636 31.85c15.186 8.866 32.733 13.554 50.863 13.554 3.398 0 6.845-.131 10.283-.393l.263-.032.23-.033c30.224-3.858 56.376-23.148 68.984-50.877l4.564-9.817a249.136 249.136 0 0040.32 3.169h.378a253.266 253.266 0 0045.738-4.729l5.713 11.755.033.05c14.676 30.026 45.376 49.086 78.801 48.906h.197c40.402-.427 75.453-28.894 84.137-68.36l.066-.33.066-.311a88.03 88.03 0 00-7.536-56.918l-10.26-21.227c33.589-36.676 58.23-82.495 67.9-133.24 4.384-22.705-.542-46.247-13.626-65.324-16.335-23.92-43.587-38.186-72.547-37.972h-.18c-31.045.377-59.61 17.303-74.829 44.359l.017-.017z"/>
          <path fill="#210784" d="M141.377 239.737l-20.308 43.67-21.21 45.573a244.52 244.52 0 0025.134 18.256A241.365 241.365 0 00181.5 373.29l20.489-43.998 19.24-41.37c-30.978-6.486-61.99-25.168-79.852-48.201v.016z"/>
          <path fill="#ff5060" d="M415.837 371.467l-23.099-47.74-42.947-88.455-62.779-129.284c-7.732-15.94-23.968-26.103-41.699-26.103-18.01 0-34.443 10.474-42.044 26.809l-61.941 132.978c17.845 23.033 48.94 41.814 79.836 48.15l20.16-43.307a5.43 5.43 0 014.909-3.136 5.422 5.422 0 014.86 3.054l20.52 42.076 41.043 84.318 20.308 41.88c7.847 16.04 24.248 26.218 42.11 26.136 21.571-.23 40.287-15.432 44.916-36.495a47.484 47.484 0 00-4.153-30.88z"/>
        </svg>
      `;
    
    case 'firefox':
      return `
        <svg width="64" height="64" viewBox="0 0 87.419 81.967">
          <defs>
            <linearGradient id="firefox-gradient" x1="70.786" y1="12.393" x2="6.447" y2="74.468" gradientUnits="userSpaceOnUse">
              <stop offset=".048" stop-color="#fff44f"/>
              <stop offset=".111" stop-color="#ffe847"/>
              <stop offset=".225" stop-color="#ffc830"/>
              <stop offset=".368" stop-color="#ff980e"/>
              <stop offset=".401" stop-color="#ff8b16"/>
              <stop offset=".462" stop-color="#ff672a"/>
              <stop offset=".534" stop-color="#ff3647"/>
              <stop offset=".705" stop-color="#e31587"/>
            </linearGradient>
          </defs>
          <path fill="url(#firefox-gradient)" d="M79.616 26.827c-1.684-4.052-5.1-8.427-7.775-9.81a40.266 40.266 0 013.925 11.764l.007.065C71.391 17.92 63.96 13.516 57.891 3.924a47.099 47.099 0 01-.913-1.484 12.24 12.24 0 01-.427-.8 7.053 7.053 0 01-.578-1.535.1.1 0 00-.088-.1.138.138 0 00-.073 0c-.005 0-.013.009-.019.01l-.028.016.015-.026c-9.735 5.7-13.038 16.252-13.342 21.53a19.387 19.387 0 00-10.666 4.11 11.587 11.587 0 00-1-.757 17.968 17.968 0 01-.109-9.473 28.705 28.705 0 00-9.329 7.21h-.018c-1.536-1.947-1.428-8.367-1.34-9.708a6.928 6.928 0 00-1.294.687 28.225 28.225 0 00-3.788 3.245 33.845 33.845 0 00-3.623 4.347v.006-.007a32.733 32.733 0 00-5.2 11.743l-.052.256a61.89 61.89 0 00-.381 2.42c0 .029-.006.056-.009.085A36.937 36.937 0 005 41.042v.2a38.759 38.759 0 0076.954 6.554c.065-.5.118-.995.176-1.5a39.857 39.857 0 00-2.514-19.47z"/>
        </svg>
      `;
    
    case 'edge':
      return `
        <svg width="64" height="64" viewBox="0 0 256 256">
          <defs>
            <linearGradient id="edge-gradient" x1="63.33" y1="84.03" x2="241.67" y2="84.03" gradientTransform="matrix(1 0 0 -1 0 266)" gradientUnits="userSpaceOnUse">
              <stop offset="0" stop-color="#0c59a4"/>
              <stop offset="1" stop-color="#114a8b"/>
            </linearGradient>
          </defs>
          <path fill="url(#edge-gradient)" d="M235.68 195.46a93.73 93.73 0 01-10.54 4.71 101.87 101.87 0 01-35.9 6.46c-47.32 0-88.54-32.55-88.54-74.32A31.48 31.48 0 01117.13 105c-42.8 1.8-53.8 46.4-53.8 72.53 0 73.88 68.09 81.37 82.76 81.37 7.91 0 19.84-2.3 27-4.56l1.31-.44a128.34 128.34 0 0066.6-52.8 4 4 0 00-5.32-5.64z" transform="translate(-4.63 -4.92)"/>
        </svg>
      `;
    
    case 'opera':
      return `
        <svg width="64" height="64" viewBox="0 0 100 100" fill="none">
          <circle cx="50" cy="50" r="40" fill="#FF1B2D"/>
          <ellipse cx="50" cy="50" rx="15" ry="30" fill="#FFF"/>
        </svg>
      `;
    
    case 'chrome':
    default:
      return `
        <svg width="64" height="64" viewBox="0 0 512 512" fill="none">
          <path fill="#FFFFFF" d="M255.73,383.71c70.3,0,127.3-56.99,127.3-127.3s-56.99-127.3-127.3-127.3s-127.3,56.99-127.3,127.3S185.42,383.71,255.73,383.71z"/>
          <defs>
            <linearGradient id="chrome-green" x1="283.2852" y1="18.9008" x2="62.8264" y2="400.7473" gradientUnits="userSpaceOnUse" gradientTransform="matrix(1 0 0 -1 0 514)">
              <stop offset="0" style="stop-color:#1E8E3E"/>
              <stop offset="1" style="stop-color:#34A853"/>
            </linearGradient>
            <linearGradient id="chrome-yellow" x1="218.5901" y1="2.3333" x2="439.0491" y2="384.1796" gradientUnits="userSpaceOnUse" gradientTransform="matrix(1 0 0 -1 0 514)">
              <stop offset="0" style="stop-color:#FCC934"/>
              <stop offset="1" style="stop-color:#FBBC04"/>
            </linearGradient>
            <linearGradient id="chrome-red" x1="35.2587" y1="353.0303" x2="476.177" y2="353.0303" gradientUnits="userSpaceOnUse" gradientTransform="matrix(1 0 0 -1 0 514)">
              <stop offset="0" style="stop-color:#D93025"/>
              <stop offset="1" style="stop-color:#EA4335"/>
            </linearGradient>
          </defs>
          <path fill="url(#chrome-green)" d="M145.48,320.08L35.26,129.17c-22.35,38.7-34.12,82.6-34.12,127.29s11.76,88.59,34.11,127.29c22.35,38.7,54.49,70.83,93.2,93.17c38.71,22.34,82.61,34.09,127.3,34.08l110.22-190.92v-0.03c-11.16,19.36-27.23,35.44-46.58,46.62c-19.35,11.18-41.3,17.07-63.65,17.07s-44.3-5.88-63.66-17.05C172.72,355.52,156.65,339.44,145.48,320.08z"/>
          <path fill="url(#chrome-yellow)" d="M365.96,320.08L255.74,510.99c44.69,0.01,88.59-11.75,127.29-34.1c38.7-22.34,70.84-54.48,93.18-93.18c22.34-38.7,34.1-82.61,34.09-127.3c-0.01-44.69-11.78-88.59-34.14-127.28H255.72l-0.03,0.02c22.35-0.01,44.31,5.86,63.66,17.03c19.36,11.17,35.43,27.24,46.61,46.59c11.18,19.35,17.06,41.31,17.06,63.66C383.03,278.77,377.14,300.72,365.96,320.08L365.96,320.08z"/>
          <path fill="#1A73E8" d="M255.73,357.21c55.66,0,100.78-45.12,100.78-100.78s-45.12-100.78-100.78-100.78s-100.78,45.12-100.78,100.78S200.07,357.21,255.73,357.21z"/>
          <path fill="url(#chrome-red)" d="M255.73,129.14h220.45C453.84,90.43,421.7,58.29,383,35.95C344.3,13.6,300.4,1.84,255.71,1.84c-44.69,0-88.59,11.77-127.29,34.12c-38.7,22.35-70.83,54.5-93.16,93.2l110.22,190.92l0.03,0.02c-11.18-19.35-17.08-41.3-17.08-63.65s5.87-44.31,17.04-63.66c11.17-19.36,27.24-35.43,46.6-46.6C211.42,135.01,233.38,129.13,255.73,129.14z"/>
        </svg>
      `;
  }
}

/**
 * Initialize the application
 */
async function initialize() {
  console.log('Mnemo UI initializing...');
  
  try {
    setupEventListeners();
    console.log('Event listeners set up');
    
    await loadVaultEntries();
    console.log('Vault entries loaded:', vaultEntries.length);
    
    renderVaultList();
    console.log('Vault list rendered');
    
    loadSettings();
    console.log('Settings loaded');
    
    updateStats();
    console.log('Stats updated');
    
    console.log('Mnemo UI initialized successfully');
    
    // Load domain rules
    await loadDomainRules();
    
    // Load learned patterns
    await loadLearnedPatterns();
  } catch (error) {
    console.error('Failed to initialize Mnemo UI:', error);
  }
}

/**
 * Setup event listeners
 */
function setupEventListeners() {
  // Tab navigation
  document.querySelectorAll('.tab-button').forEach(button => {
    button.addEventListener('click', () => switchTab(button.dataset.tab));
  });
  
  // Search
  document.getElementById('searchInput').addEventListener('input', handleSearch);
  
  // Sort
  document.getElementById('sortSelect').addEventListener('change', (e) => {
    sortBy = e.target.value;
    renderVaultList();
  });
  
  // Settings
  document.getElementById('archiveThreshold').addEventListener('input', (e) => {
    document.getElementById('archiveThresholdValue').textContent = e.target.value;
  });
  
  document.getElementById('purgeDataBtn').addEventListener('click', handlePurgeData);
  document.getElementById('clearVaultBtn').addEventListener('click', handleClearVault);
  document.getElementById('clearOldEntriesBtn').addEventListener('click', handleClearOldEntries);
  document.getElementById('clearHistoryBtn').addEventListener('click', handleClearHistory);
  
  // Domain rules
  document.getElementById('addRuleBtn').addEventListener('click', addDomainRule);
  document.getElementById('newRuleType').addEventListener('change', (e) => {
    const valueInput = document.getElementById('newRuleValue');
    valueInput.style.display = e.target.value === 'never' ? 'none' : 'block';
    if (e.target.value === 'threshold') {
      valueInput.placeholder = 'Threshold (0-30)';
      valueInput.max = 30;
    } else if (e.target.value === 'inactive') {
      valueInput.placeholder = 'Minutes (0-180)';
      valueInput.max = 180;
    }
  });
  
  // Range slider updates
  const archiveThresholdSlider = document.getElementById('archiveThreshold');
  archiveThresholdSlider.addEventListener('input', (e) => {
    updateRangeSlider('archiveThreshold');
  });
  
  // Save settings on change
  const settingInputs = document.querySelectorAll('#settingsPanel input, #settingsPanel select, #settingsPanel textarea');
  settingInputs.forEach(input => {
    input.addEventListener('change', saveSettings);
  });
}

/**
 * Switch between tabs
 */
function switchTab(tabName) {
  currentTab = tabName;
  
  // Update button states
  document.querySelectorAll('.tab-button').forEach(button => {
    button.classList.toggle('active', button.dataset.tab === tabName);
  });
  
  // Update panel visibility
  document.querySelectorAll('.panel').forEach(panel => {
    panel.classList.remove('active');
  });
  
  document.getElementById(`${tabName}Panel`).classList.add('active');
  
  // Load data for the selected tab
  if (tabName === 'timeline') {
    loadTimeline();
  }
}

/**
 * Load vault entries from storage
 */
async function loadVaultEntries() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_VAULT_ENTRIES'
    });
    
    vaultEntries = response?.entries || [];
    
  } catch (error) {
    console.error('Failed to load vault entries:', error);
    vaultEntries = [];
  }
}

/**
 * Render vault list
 */
function renderVaultList() {
  const container = document.getElementById('vaultList');
  const emptyState = document.getElementById('emptyState');
  
  let displayEntries = [...vaultEntries];
  
  // Filter by search query
  if (searchQuery) {
    displayEntries = displayEntries.filter(entry =>
      entry.title?.toLowerCase().includes(searchQuery) ||
      entry.url?.toLowerCase().includes(searchQuery) ||
      entry.domain?.toLowerCase().includes(searchQuery) ||
      entry.textExtract?.toLowerCase().includes(searchQuery)
    );
  }
  
  // Sort
  if (sortBy === 'recent') {
    displayEntries.sort((a, b) => b.archivedAt - a.archivedAt);
  } else if (sortBy === 'score') {
    // If we have search scores, sort by those, otherwise by original score
    if (displayEntries.some(e => e.searchScore !== undefined)) {
      displayEntries.sort((a, b) => (b.searchScore || 0) - (a.searchScore || 0));
    } else {
      displayEntries.sort((a, b) => (b.score || 0) - (a.score || 0));
    }
  } else if (sortBy === 'domain') {
    displayEntries.sort((a, b) => (a.domain || '').localeCompare(b.domain || ''));
  }
  
  if (displayEntries.length === 0) {
    container.style.display = 'none';
    emptyState.style.display = 'flex';
  } else {
    container.style.display = 'block';
    emptyState.style.display = 'none';
    container.innerHTML = displayEntries.map(entry => createVaultCard(entry)).join('');
    
    // Attach event listeners
    container.querySelectorAll('.restore-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const vaultId = e.currentTarget.dataset.id;
        restoreVaultEntry(vaultId);
      });
    });
    
    container.querySelectorAll('.delete-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const vaultId = e.currentTarget.dataset.id;
        deleteVaultEntry(vaultId);
      });
    });
  }
}
/**
 * Create HTML for vault card
 */
function createVaultCard(entry) {
  const timeAgo = formatTimeAgo(entry.archivedAt);
  const domain = entry.domain || new URL(entry.url).hostname;
  
  return `
    <div class="vault-card">
      ${entry.screenshot ? `
        <div class="vault-thumbnail">
          <img src="${entry.screenshot}" alt="${entry.title}">
        </div>
      ` : `
        <div class="vault-thumbnail vault-thumbnail-empty">
          ${getBrowserLogoSVG()}
        </div>
      `}
      
      <div class="vault-content">
        <h3 class="vault-title">${escapeHtml(entry.title || 'Untitled')}</h3>
        <p class="vault-domain">
          ${entry.favIconUrl ? `<img src="${entry.favIconUrl}" class="favicon" alt="" onerror="this.style.display='none'">` : ''}
          ${escapeHtml(domain)}
        </p>
        
        <div class="vault-meta">
          <span class="meta-item">
            <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <circle cx="12" cy="12" r="10"/>
              <polyline points="12 6 12 12 16 14"/>
            </svg>
            ${timeAgo}
          </span>
        </div>
      </div>
      
      <div class="vault-actions">
        <button class="action-btn restore-btn" data-id="${entry.id}" title="Restore">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polyline points="23 4 23 10 17 10"/>
            <path d="M20.49 15a9 9 0 1 1-2.12-9.36L23 10"/>
          </svg>
        </button>
        <button class="action-btn delete-btn" data-id="${entry.id}" title="Delete">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polyline points="3 6 5 6 21 6"/>
            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
          </svg>
        </button>
      </div>
    </div>
  `;
}

/**
 * Sort vault entries
 */
function sortVaultEntries(entries) {
  switch (sortBy) {
    case 'recent':
      return entries.sort((a, b) => b.archivedAt - a.archivedAt);
      
    case 'score':
      return entries.sort((a, b) => (b.score || 0) - (a.score || 0));
      
    case 'domain':
      return entries.sort((a, b) => (a.domain || '').localeCompare(b.domain || ''));
      
    default:
      return entries;
  }
}

/**
 * Handle search input
 */
async function handleSearch(e) {
  searchQuery = e.target.value.toLowerCase().trim();
  
  if (searchQuery.length === 0) {
    // Reset to all entries
    await loadVaultEntries();
    renderVaultList();
    return;
  }
  
  if (searchQuery.length < 2) {
    return; // Wait for at least 2 characters
  }
  
  // Use backend TF-IDF search
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'SEARCH_VAULT',
      query: searchQuery
    });
    
    vaultEntries = response?.results || [];
    renderVaultList();
    
  } catch (error) {
    console.error('Search failed:', error);
  }
}

/**
 * Restore vault entry
 */
async function restoreVaultEntry(vaultId) {
  try {
    await chrome.runtime.sendMessage({
      type: 'RESTORE_FROM_VAULT',
      vaultId
    });
    
    // Remove from current list
    vaultEntries = vaultEntries.filter(e => e.id !== vaultId);
    renderVaultList();
    updateStats();
    
  } catch (error) {
    console.error('Restore failed:', error);
    alert('Failed to restore tab');
  }
}

/**
 * Delete vault entry
 */
async function deleteVaultEntry(vaultId) {
  if (!confirm('Are you sure you want to delete this archived tab?')) {
    return;
  }
  
  try {
    await chrome.runtime.sendMessage({
      type: 'DELETE_VAULT_ENTRY',
      vaultId
    });
    
    vaultEntries = vaultEntries.filter(e => e.id !== vaultId);
    renderVaultList();
    updateStats();
    
  } catch (error) {
    console.error('Delete failed:', error);
    alert('Failed to delete entry');
  }
}

/**
 * Load timeline
 */
async function loadTimeline() {
  const container = document.getElementById('timelineList');
  
  // Group vault entries by date
  const grouped = groupByDate(vaultEntries);
  
  container.innerHTML = Object.entries(grouped)
    .map(([date, entries]) => `
      <div class="timeline-group">
        <h3 class="timeline-date">${date}</h3>
        <div class="timeline-items">
          ${entries.map(entry => `
            <div class="timeline-item">
              <span class="timeline-time">${formatTime(entry.archivedAt)}</span>
              <span class="timeline-title">${escapeHtml(entry.title || 'Untitled')}</span>
              <span class="timeline-domain">${escapeHtml(entry.domain || '')}</span>
            </div>
          `).join('')}
        </div>
      </div>
    `).join('');
}

/**
 * Group entries by date
 */
function groupByDate(entries) {
  const groups = {};
  
  entries.forEach(entry => {
    const date = new Date(entry.archivedAt);
    const dateKey = date.toLocaleDateString('en-US', { 
      month: 'long', 
      day: 'numeric', 
      year: 'numeric' 
    });
    
    if (!groups[dateKey]) {
      groups[dateKey] = [];
    }
    
    groups[dateKey].push(entry);
  });
  
  return groups;
}

/**
 * Load settings
 */
async function loadSettings() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_SETTINGS'
    });
    
    const settings = response?.settings || {};
    
    document.getElementById('archiveThreshold').value = settings.archiveThreshold || 10;
    document.getElementById('maxActiveTabs').value = settings.maxActiveTabs || 20;
    document.getElementById('minInactiveMinutes').value = settings.minInactiveMinutes || 30;
    document.getElementById('autoArchive').checked = settings.autoArchive ?? true;
    document.getElementById('enableDuplicateDetection').checked = settings.enableDuplicateDetection ?? true;
    document.getElementById('captureScreenshots').checked = settings.captureScreenshots ?? true;
    document.getElementById('vaultRetention').value = settings.vaultRetention || 90;
    document.getElementById('enableMLPersonalization').checked = settings.enableMLPersonalization ?? true;
    document.getElementById('enableSmartStartup').checked = settings.enableSmartStartup ?? false;
    
    // Update range slider display
    updateRangeSlider('archiveThreshold');
    
  } catch (error) {
    console.error('Load settings failed:', error);
  }
}

// Update range slider background based on value
function updateRangeSlider(id) {
  const slider = document.getElementById(id);
  const output = document.getElementById(id + 'Value');
  const min = slider.min || 0;
  const max = slider.max || 100;
  const value = slider.value;
  
  // Update output display
  if (output) {
    output.textContent = value;
  }
  
  // Calculate percentage
  const percentage = ((value - min) / (max - min)) * 100;
  
  // Update background gradient
  slider.style.background = `linear-gradient(to right, var(--accent-primary) 0%, var(--accent-primary) ${percentage}%, var(--bg-tertiary) ${percentage}%, var(--bg-tertiary) 100%)`;
}

/**
 * Save settings
 */
async function saveSettings() {
  const settings = {
    archiveThreshold: parseInt(document.getElementById('archiveThreshold').value),
    maxActiveTabs: parseInt(document.getElementById('maxActiveTabs').value),
    minInactiveMinutes: parseInt(document.getElementById('minInactiveMinutes').value),
    autoArchive: document.getElementById('autoArchive').checked,
    enableDuplicateDetection: document.getElementById('enableDuplicateDetection').checked,
    captureScreenshots: document.getElementById('captureScreenshots').checked,
    vaultRetention: parseInt(document.getElementById('vaultRetention').value),
    enableMLPersonalization: document.getElementById('enableMLPersonalization').checked,
    enableSmartStartup: document.getElementById('enableSmartStartup').checked
  };
  
  try {
    await chrome.runtime.sendMessage({
      type: 'SAVE_SETTINGS',
      settings
    });
  } catch (error) {
    console.error('Failed to save settings:', error);
  }
}

/**
 * Handle purge data
 */
async function handlePurgeData() {
  const confirmed = confirm(
    'This will permanently delete ALL archived tabs and data. This action cannot be undone. Are you sure?'
  );
  
  if (!confirmed) return;
  
  const doubleConfirmed = confirm('Are you ABSOLUTELY sure? This will delete everything.');
  
  if (!doubleConfirmed) return;
  
  try {
    await chrome.runtime.sendMessage({
      type: 'PURGE_ALL_DATA'
    });
    
    vaultEntries = [];
    renderVaultList();
    updateStats();
    
    alert('All data has been purged');
    
  } catch (error) {
    console.error('Purge failed:', error);
    alert('Failed to purge data');
  }
}

/**
 * Handle clear vault (delete all archives)
 */
async function handleClearVault() {
  const confirmed = confirm(
    'This will delete ALL archived tabs from the vault. This action cannot be undone. Continue?'
  );
  
  if (!confirmed) return;
  
  try {
    await chrome.runtime.sendMessage({
      type: 'CLEAR_VAULT'
    });
    
    vaultEntries = [];
    renderVaultList();
    updateStats();
    
    alert('Vault cleared successfully');
    
  } catch (error) {
    console.error('Clear vault failed:', error);
    alert('Failed to clear vault');
  }
}

/**
 * Handle clear old entries
 */
async function handleClearOldEntries() {
  const confirmed = confirm(
    'This will delete vault entries older than your retention period. Continue?'
  );
  
  if (!confirmed) return;
  
  try {
    await chrome.runtime.sendMessage({
      type: 'CLEAR_OLD_ENTRIES'
    });
    
    await loadVaultEntries();
    renderVaultList();
    updateStats();
    
    alert('Old entries cleared successfully');
    
  } catch (error) {
    console.error('Clear old entries failed:', error);
    alert('Failed to clear old entries');
  }
}

/**
 * Handle clear history/signals
 */
async function handleClearHistory() {
  const confirmed = confirm(
    'This will clear all captured interaction signals and timeline data. Continue?'
  );
  
  if (!confirmed) return;
  
  try {
    await chrome.runtime.sendMessage({
      type: 'CLEAR_HISTORY'
    });
    
    alert('History cleared successfully');
    
  } catch (error) {
    console.error('Clear history failed:', error);
    alert('Failed to clear history');
  }
}

/**
 * Load domain rules
 */
async function loadDomainRules() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_DOMAIN_RULES'
    });
    
    const rules = response?.rules || [];
    renderDomainRules(rules);
  } catch (error) {
    console.error('Failed to load domain rules:', error);
  }
}

/**
 * Render domain rules list
 */
function renderDomainRules(rules) {
  const list = document.getElementById('domainRulesList');
  
  if (!rules || rules.length === 0) {
    list.innerHTML = '<div class="empty-rules">No domain rules yet. Add one below.</div>';
    return;
  }
  
  list.innerHTML = rules.map(rule => {
    let ruleText = '';
    if (rule.type === 'never') {
      ruleText = 'Never archive';
    } else if (rule.type === 'threshold') {
      ruleText = `Archive below ${rule.value}`;
    } else if (rule.type === 'inactive') {
      ruleText = `Archive after ${rule.value}min inactive`;
    }
    
    return `
      <div class="domain-rule-item">
        <div class="rule-domain">${escapeHtml(rule.domain)}</div>
        <div class="rule-description">${ruleText}</div>
        <button class="rule-delete" data-domain="${escapeHtml(rule.domain)}">
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M3 6h18M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
          </svg>
        </button>
      </div>
    `;
  }).join('');
  
  // Attach delete handlers
  list.querySelectorAll('.rule-delete').forEach(btn => {
    btn.addEventListener('click', async (e) => {
      const domain = e.currentTarget.dataset.domain;
      await deleteDomainRule(domain);
    });
  });
}

/**
 * Add new domain rule
 */
async function addDomainRule() {
  const domain = document.getElementById('newRuleDomain').value.trim();
  const type = document.getElementById('newRuleType').value;
  const value = parseInt(document.getElementById('newRuleValue').value) || 0;
  
  if (!domain) {
    alert('Please enter a domain');
    return;
  }
  
  // Validate domain format (basic)
  if (!domain.includes('.')) {
    alert('Please enter a valid domain (e.g., example.com)');
    return;
  }
  
  if (type !== 'never' && !value) {
    alert('Please enter a value');
    return;
  }
  
  try {
    await chrome.runtime.sendMessage({
      type: 'ADD_DOMAIN_RULE',
      rule: { domain, type, value: type === 'never' ? null : value }
    });
    
    // Clear inputs
    document.getElementById('newRuleDomain').value = '';
    document.getElementById('newRuleValue').value = '';
    
    // Reload rules
    await loadDomainRules();
  } catch (error) {
    console.error('Failed to add rule:', error);
    alert('Failed to add rule');
  }
}

/**
 * Delete domain rule
 */
async function deleteDomainRule(domain) {
  try {
    await chrome.runtime.sendMessage({
      type: 'DELETE_DOMAIN_RULE',
      domain
    });
    
    await loadDomainRules();
  } catch (error) {
    console.error('Failed to delete rule:', error);
    alert('Failed to delete rule');
  }
}

/**
 * Update statistics
 */
async function updateStats() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_STORAGE_STATS'
    });
    
    const stats = response?.stats;
    
    if (stats) {
      document.getElementById('vaultCount').textContent = `${stats.vaultCount} Tabs archived`;
      document.getElementById('archivedCount').textContent = stats.vaultCount;
      document.getElementById('storageUsed').textContent = 
        `${formatBytes(stats.storageUsed)} / ${formatBytes(stats.storageQuota)} (${stats.usagePercent}%)`;
    }
    
  } catch (error) {
    console.error('Failed to update stats:', error);
  }
}

/**
 * Load and display learned patterns
 */
async function loadLearnedPatterns() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_LEARNED_PATTERNS'
    });
    
    const patterns = response?.patterns || [];
    const list = document.getElementById('learnedPatternsList');
    
    if (patterns.length === 0) {
      list.innerHTML = '<div class="empty-patterns">Learning your browsing habits... Check back after a week of usage.</div>';
      return;
    }
    
    list.innerHTML = patterns.slice(0, 10).map(p => `
      <div class="pattern-item">
        <div class="pattern-domain">${escapeHtml(p.domain)}</div>
        <div class="pattern-stats">
          <div class="pattern-importance" style="width: ${p.importance}%">
            <span>${p.importance}%</span>
          </div>
          <div class="pattern-meta">
            ${p.visitCount} visits • ${formatDuration(p.totalTime)} • 
            ${p.peakDays.join(', ')} @ ${p.peakHours.map(h => h + 'h').join(', ')}
          </div>
        </div>
      </div>
    `).join('');
    
  } catch (error) {
    console.error('Failed to load learned patterns:', error);
  }
}

function formatDuration(ms) {
  const hours = Math.floor(ms / 3600000);
  const minutes = Math.floor((ms % 3600000) / 60000);
  if (hours > 0) return `${hours}h ${minutes}m`;
  return `${minutes}m`;
}

/**
 * Utility functions
 */
function formatTimeAgo(timestamp) {
  const seconds = Math.floor((Date.now() - timestamp) / 1000);
  
  if (seconds < 60) return 'Just now';
  if (seconds < 3600) return `${Math.floor(seconds / 60)}m ago`;
  if (seconds < 86400) return `${Math.floor(seconds / 3600)}h ago`;
  if (seconds < 604800) return `${Math.floor(seconds / 86400)}d ago`;
  
  return new Date(timestamp).toLocaleDateString();
}

function formatTime(timestamp) {
  return new Date(timestamp).toLocaleTimeString('en-US', { 
    hour: 'numeric', 
    minute: '2-digit' 
  });
}

function formatBytes(bytes) {
  if (bytes === 0) return '0 B';
  const k = 1024;
  const sizes = ['B', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return `${(bytes / Math.pow(k, i)).toFixed(2)} ${sizes[i]}`;
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initialize);
} else {
  initialize();
}
